﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;

    using Domain.Helpers;

    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;

    using Newtonsoft.Json;

    using Shared.DataFilters;
    using Shared.UserModels.PayU;

    using Utilities;

    using Hims.Api.Senders;
    using Hims.Domain.Services;
    using Hims.Shared.EntityModels;
    using Hims.Shared.UserModels.Filters;

    /// <inheritdoc />
    [Authorize]
    [Route("api/payU")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PayUController : BaseController
    {
        /// <summary>
        /// The payU helper.
        /// </summary>
        private readonly IPayUHelper payUHelper;

        /// <summary>
        /// The patient device details services.
        /// </summary>
        private readonly IPayUService payUServices;

        /// <summary>
        /// The email sender.
        /// </summary>
        private readonly IEmailSender emailSender;

        /// <inheritdoc />
        public PayUController(IPayUHelper payUHelper, IPayUService payUServices, IEmailSender emailSender)
        {
            this.payUHelper = payUHelper;
            this.payUServices = payUServices;
            this.emailSender = emailSender;
        }

        /// <summary>
        /// The fetch appointment transaction payU status details.
        /// </summary>
        /// <param name="model">
        /// The appointment transaction filter model.
        /// </param>
        /// <returns>
        /// Appointment  transaction payU status details.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Appointment transaction payU details.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetchStatus")]
        [ProducesResponseType(typeof(List<PaymentStatusModel>), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchTransactionStatusAsync([FromBody] AppointmentTransactionFilterModel model)
        {
            model = (AppointmentTransactionFilterModel)EmptyFilter.Handler(model);
            var response = await this.payUHelper.FetchStatusAsync(model.TransactionId);
            return this.Success(JsonConvert.DeserializeObject<PaymentStatusModel>(response.Content));
        }

        /// <summary>
        /// The refund appointment transaction payU .
        /// </summary>
        /// <param name="model">
        /// The appointment transaction filter model.
        /// </param>
        /// <returns>
        /// Appointment  transaction refund  details.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Appointment transaction payU details.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("refund")]
        [ProducesResponseType(typeof(List<PayURefundModel>), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> RefundAsync([FromBody] RefundModel model)
        {
            model = (RefundModel)EmptyFilter.Handler(model);
            var response = await this.payUHelper.RefundAsync(model.PaymentId, model.RefundAmount);
            var item = JsonConvert.DeserializeObject<PayURefundModel>(response.Content);
            if (!response.IsSuccessful || !(item.Status >= 0))
            {
                return this.Conflict(item.Message);
            }

            model.RefundStatus = item.Message;
            if (item.Result != null)
            {
                model.RefundTransactionId = (int)item.Result;
            }

            var refundId = await this.payUServices.AddRefundAsync(model);
            switch (refundId)
            {
                case -1:
                    return this.Conflict("Refund for this payment already done .");
                case 0:
                    return this.ServerError();
                default:
                    return this.Success(item);
            }
        }

        /// <summary>
        /// The fetch appointment transaction payU status details.
        /// </summary>
        /// <param name="model">
        /// The appointment transaction filter model.
        /// </param>
        /// <returns>
        /// Appointment  transaction payU status details.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Appointment transaction payU details.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetchRefundStatus")]
        [ProducesResponseType(typeof(List<RefundModel>), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRefundStatusAsync([FromBody] AppointmentTransactionFilterModel model)
        {
            model = (AppointmentTransactionFilterModel)EmptyFilter.Handler(model);
            var response = await this.payUHelper.FetchRefundStatusAsync(model.PaymentId);
            return this.Success(response.Content);
        }

        /// <summary>
        /// The payout authentication.
        /// </summary>
        /// <returns>
        ///  Payout token details
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Appointment transaction payU details.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("authenticate")]
        [ProducesResponseType(typeof(PayoutModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> PayoutAuthenticateAsync()
        {
            var response = await this.payUHelper.PayoutAuthentication();
            return this.Success(JsonConvert.DeserializeObject<PayoutModel>(response.Content));
        }

        /// <summary>
        /// The payout account details async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("getAccountDetail")]
        [ProducesResponseType(typeof(PayoutModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> PayoutAccountDetailsAsync([FromBody] PayoutModel model)
        {
            model = (PayoutModel)EmptyFilter.Handler(model);
            var response = await this.payUHelper.PayoutAccountDetails(model);
            return this.Success(JsonConvert.DeserializeObject<PayoutResponseModel>(response.Content));
        }

        /// <summary>
        /// The payout account details async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("smartSend")]
        [ProducesResponseType(typeof(PayoutResponseModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<bool> PayoutSmartSendAsync([FromBody] PayoutModel model)
        {
            model = (PayoutModel)EmptyFilter.Handler(model);
            model.MerchantRefId = Guid.NewGuid().ToString().Substring(0, 8);
            var response = await this.payUHelper.PayoutSmartSendAsync(model);
            var result = JsonConvert.DeserializeObject<PayoutResponseModel>(response.Content);
            if (result.Data != null && !string.IsNullOrEmpty(result.Data.LinkId))
            {
                return await this.emailSender.SendPaymentMailAsync(model.Email, model.UserName, result.Data.Link);
            }

            return false;
        }

        /// <summary>
        /// The payout account details async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("payment")]
        [ProducesResponseType(typeof(PaymentStatusModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<PaymentStatusModel> PayoutPaymentAsync([FromBody] PayoutModel model)
        {
            model = (PayoutModel)EmptyFilter.Handler(model);
            model.MerchantRefId = Guid.NewGuid().ToString().Substring(0, 8);
            var response = await this.payUHelper.PayoutPaymentAsync(model);
            return JsonConvert.DeserializeObject<dynamic>(response.Content);
        }

        /// <summary>
        /// The payout account details async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("addWebHook")]
        [ProducesResponseType(typeof(PaymentStatusModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<PaymentStatusModel> AddWebHookAsync([FromBody] PayoutModel model)
        {
            model = (PayoutModel)EmptyFilter.Handler(model);
            model.MerchantRefId = Guid.NewGuid().ToString().Substring(0, 8);
            var response = await this.payUHelper.AddWebHook(model);
            return JsonConvert.DeserializeObject<PaymentStatusModel>(response.Content);
        }

        /// <summary>
        /// The payout account details async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("deleteWebHook")]
        [ProducesResponseType(typeof(PaymentStatusModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<PaymentStatusModel> DeleteWebHookAsync([FromBody] PayoutModel model)
        {
            model = (PayoutModel)EmptyFilter.Handler(model);
            var response = await this.payUHelper.DeleteWebHook(model);
            return JsonConvert.DeserializeObject<PaymentStatusModel>(response.Content);
        }

        /// <summary>
        /// The payout account details async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("WebHook")]
        [AllowAnonymous]
        public async Task WebHookAsync([FromBody] WebHook model)
        {
            if (!string.IsNullOrEmpty(model.Event))
            {
                var data = new WebHookModel
                {
                    WebHookData = JsonConvert.SerializeObject(model),
                    WebHookEvent = model.Event
                };
                await this.payUServices.AddWebHookAsync(data);
            }
        }

        /// <summary>
        /// The fetch appointment transaction payU status details.
        /// </summary>
        /// <param name="model">
        /// The appointment transaction filter model.
        /// </param>
        /// <returns>
        /// Appointment  transaction payU status details.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Appointment transaction payU details.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetchPayoutTransfers")]
        [ProducesResponseType(typeof(List<PaymentStatusModel>), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPayoutTransactionsAsync([FromBody] PayoutModel model)
        {
            model = (PayoutModel)EmptyFilter.Handler(model);
            var response = await this.payUHelper.FetchPayoutTransfers(model);
            return this.Success(JsonConvert.DeserializeObject<PaymentStatusModel>(response.Content));
        }
    }
}
